/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

#if defined(_MSC_VER)
# pragma data_seg("mini_api_c","FAR_DATA")
#elif defined(__BORLANDC__)
# pragma option -dc
#endif /* _MSC_VER etc. */


/* main api file for static and board related functions */
#include <typedefs.h>

#include <iocommon.h>
#include <mini_api.h>
#include <regconst.h>
#include <regxdir.h>
#include <regx10.h>
#include <regx11.h>
#include <regx14.h>
#include <regx1a.h>
#include <regx1c.h>
#include <regx22.h>
#include <regx23.h>
#include <session.h>
#include "version.h"

#include <sst_prop.h>

#include "pci.h"
#include "timeout.h"


#define RETRY_COUNT_AFTER_BOOT  300

/*   default for        block size, fast host threshold */
#define B_DEF_CAPIPROP  { 0x1F000UL, 0 }
#define FOUR_CP_INITS B_DEF_CAPIPROP, B_DEF_CAPIPROP,   \
      B_DEF_CAPIPROP, B_DEF_CAPIPROP
      
b_int32 best_capi_prop [MAXHANDLES + 1][B_CAPIPROP_SIZE] = {
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, 
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS,
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, 
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS,
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, 
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS,
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, 
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS,
  B_DEF_CAPIPROP
};


/* -----------------------------------------------------------------
 * CPU Port Programming Functions moved to cpuport.c
 * ----------------------------------------------------------------- */

/* -----------------------------------------------------------------
 * PME Bit access Functions
 * ----------------------------------------------------------------- */

b_errtype EXPORT BestPMEWrite(
    b_handletype handle,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestPMEWrite [pmewrite]");
  B_TRY_VARS_NO_PROG;
  b_int8 data = (b_int8) value;

  B_TRY_BEGIN
  {

    /* TODO; verify that this is the correct license check */
    B_TRY_LICENSE(B_CAPABILITY_CAPI);
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    B_TRY_FAIL(BestIsCompact(handle) ? B_E_NOT_COMPACT : B_E_OK);
    B_TRY_FCT_PARAM(1, value > 1UL);

    B_TRY(BBCSendByte(handle, CMD_PME_WRITE, &data));
  }

  B_ERRETURN(B_TRY_RET);
}

b_errtype EXPORT BestPMERead(
    b_handletype handle,
    b_int32 * value_ptr
)
{
  B_DECLARE_FUNCNAME("BestPMERead [pmeread]");

  B_TRY_VARS_NO_PROG;
  b_int8 ch = 0;

  B_FCT_PARAM_NULL_POINTER_CHK(value_ptr);

  B_TRY_BEGIN
  {
    /* TODO; verify that this is the correct license check */
    B_TRY_LICENSE(B_CAPABILITY_CAPI);
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    B_TRY_FAIL(BestIsCompact(handle) ? B_E_NOT_COMPACT : B_E_OK);

    B_TRY(BBCReceiveByte(handle, CMD_PME_READ, &ch));
    *value_ptr = (b_int32) ch;
  }

  B_ERRETURN(B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * Interrupt Generation Function
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestInterruptGenerate(b_handletype handle, b_int32 pci_int)
{
# define INTR_MASK_TEST   (b_int8)(B_INTA | B_INTB | B_INTC | B_INTD)

  B_DECLARE_FUNCNAME("BestInterruptGenerate [intgen]");

  b_int8 intr_mask8 = (b_int8) pci_int;
  B_FCT_PARAM_CHK_R(2, (intr_mask8 & ~INTR_MASK_TEST),
    "mask has illegal bits set");

  if (Best16BitRegisterFile(handle))
  {
    return BBCSendByte(handle, CMD_INTR_GENERATE, &intr_mask8);
  }
  else
  {
    return BestBasicBlockWrite(handle, INTR_GENERATE, (b_int8 *) & intr_mask8, 1UL);
  }
}



#ifndef E2921A_MIN

/* -----------------------------------------------------------------
 * PCI Config Mailbox Register Functions
 * moved to mailbox.c (HL, 15.9.1999)
 * ----------------------------------------------------------------- */
/* --------------------------------------------------------------------------
 * Mailbox Register Functions
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestMailboxReceiveRegRead(
    b_handletype handle,
    b_int32 * value,
    b_int32 * status
)
{
  B_TRY_VARS_NO_PROG;
  b_int8 data8;
  b_int8 buf[OUT_MAILBOX_READ];
  b_int8ptr bufptr;
  *value = 0xffffffffUL;        /* init for failure */
  *status = 0;

  B_TRY_BEGIN
  {
    BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PORT_COUNT + 1UL);
    BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)handle_array[handle].port);
    B_TRY_FAIL(((handle_array[handle].port == B_PORT_PCI_CONF) ||
        (handle_array[handle].port == B_PORT_PCI_IO)) ?
      B_E_WRONG_PORT : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      b_int16 outsize = OUT_MAILBOX_READ;
      B_TRY(BestBasicCommand(handle, CMD_MAILBOX_READ,
          NULL, IN_MAILBOX_READ, buf, &outsize));

      bufptr = BestStream2Long(value, buf, 1UL);
      (void) BestByteCopy(&data8, bufptr, 1UL);
      *status = data8;          /* msg not previously read */
    }
    else
    {
      B_TRY(BestBasicBlockRead(handle, MAILBOX_ACCESS, &data8, 1UL));
      *status = 0x1;            /* msg not previously read */
      *value = (b_int32) data8;
    }
  }

  B_TRY_CATCH
  {
    *status = 0;

    if (B_TRY_RET == B_E_FUNC)
      B_TRY_RET = B_E_OK;       /* this is for 25 return mechanism */
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestMailboxSendRegWrite(
    b_handletype handle,
    b_int32 value,
    b_int32 * status
)
{
  B_TRY_VARS_NO_PROG;
  b_int8 data8;
  b_int8 buf[sizeof(b_int32)];
  B_TRY_BEGIN
  {
    BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PORT_COUNT + 1UL);
    BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)handle_array[handle].port);
    B_TRY_FAIL(((handle_array[handle].port == B_PORT_PCI_CONF) ||
        (handle_array[handle].port == B_PORT_PCI_IO)) ?
      B_E_WRONG_PORT : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      b_int16 outsize = OUT_MAILBOX_WRITE;
      (void) BestLong2Stream(buf, &value, 1UL);
      B_TRY(BestBasicCommand(handle, CMD_MAILBOX_WRITE,
          buf, IN_MAILBOX_WRITE, &data8, &outsize));
      *status = data8;          /* msg not previously read */
    }
    else
    {
      *status = 0x01;
      data8 = (b_int8) value;
      B_TRY(BestBasicBlockWrite(handle, MAILBOX_ACCESS, &data8, 1UL));
    }
  }

  B_TRY_CATCH
  {
    *status = 0;

    if (B_TRY_RET == B_E_FUNC)
      B_TRY_RET = B_E_OK;       /* this is for 25 return mechanism */
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* E2921A_MIN */

/* -----------------------------------------------------------------
 * Hex Dixplay Functions
 * ----------------------------------------------------------------- */

b_errtype EXPORT BestDisplayPropSet(
    b_handletype handle,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestDisplayPropSet [dprpset]");

  b_int8 help = (b_int8) value;
  b_errtype err;
  B_FCT_PARAM_CHK(2, ((value != B_DISP_USER) && (value != B_DISP_CARD)));

  if (Best16BitRegisterFile(handle))
  {
    err = BestBasicCommand(handle, CMD_HEX_DISPLAY, &help, IN_HEX_DISPLAY, NULL, NULL);
  }
  else
  {
    err = BestBasicBlockWrite(handle, HEX_DISP_MODE, &help, 1UL);
  }

  B_ERRETURN(err);
}


b_errtype EXPORT BestDisplayWrite(
    b_handletype handle,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestDisplayWrite [dwrite]");

  b_int8 help = (b_int8) value;
  b_errtype err;
  B_FCT_PARAM_CHK(2, value > 0xff);

  if (Best16BitRegisterFile(handle))
  {
    err = BestBasicCommand(handle, CMD_HEX_SHOW, &help, IN_HEX_SHOW, NULL, NULL);
  }
  else
  {
    err = BestBasicBlockWrite(handle, HEX_DISP_DATA, (b_int8 *) & value, 1UL);
  }

  B_ERRETURN(err);
}


#ifndef E2921A_MIN

/* -----------------------------------------------------------------
 * Power Up Behaviour Functions
 * ----------------------------------------------------------------- */

b_errtype EXPORT BestPowerUpPropSet(
    b_handletype handle,
    b_puproptype pu_prop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestPowerUpPropSet [puprpset]");
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* no license checking */

    B_TRY(BestParamCheck(handle, B_PARAM_PUINFO, (b_int32)pu_prop, value));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongSet(handle, CMD_PU_PROP_SET,
          (b_int8) pu_prop, value));
    }
    else
    {
      b_int32 regnr;
      b_int8 zw;
      zw = (b_int8) value;

      switch (pu_prop)
      {
      case B_PU_OBSRUNMODE:
        regnr = OBS_PU_MODE;
        break;

      case B_PU_TRCRUNMODE:
        regnr = ANA_PU_MODE;
        break;

      case B_PU_SSTRUNMODE:
        regnr = SST_PU_MODE;
        break;

      case B_PU_CONFRESTORE:
        regnr = CONF_PUMODE;
        break;

      default:
        B_FCT_PARAM_ERROR(2, "Invalid Property");
      } /*lint !e788 not all enums used */

      B_TRY(BestBasicBlockWrite(handle, regnr, &zw, 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestPowerUpPropGet(
    b_handletype handle,
    b_puproptype pu_prop,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestPowerUpPropGet [puprpget]");

  B_TRY_VARS_NO_PROG;

  B_FCT_PARAM_NULL_POINTER_CHK(value);

  B_TRY_BEGIN
  {
    B_TRY(BestParamInfoGet(handle, B_PARAM_PUINFO,
        (b_int32)pu_prop, NULL, (b_int32)B_ENUM_SEARCH));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongGet(handle, CMD_PU_PROP_GET,
          (b_int8) pu_prop, value));
    }
    else
    {
      b_int32 regnr;
      b_int8 zw;
      switch (pu_prop)
      {
      case B_PU_OBSRUNMODE:
        regnr = OBS_PU_MODE;
        break;

      case B_PU_TRCRUNMODE:
        regnr = ANA_PU_MODE;
        break;

      case B_PU_SSTRUNMODE:
        regnr = SST_PU_MODE;
        break;

      case B_PU_CONFRESTORE:
        regnr = CONF_PUMODE;
        break;

      default:
        B_FCT_PARAM_ERROR(2, "Invalid Property");
      } /*lint !e788 not all enums used */

      B_TRY(BestBasicBlockRead(handle, regnr, &zw, 1UL));
      *value = (b_int32) zw;
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* sets all power up properies to their specified defautl value */
b_errtype EXPORT BestPowerUpPropDefaultSet(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;
  b_int32 i;
  b_generic_infotype *GenInfo;
  b_param_infotype *ParamInfo;
  B_TRY_BEGIN
  {
    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_PUINFO, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, B_PARAM_PUINFO,
          i, &ParamInfo, (b_int32)B_INDEX_SEARCH));

      /* Set it to default */
      B_TRY(BestPowerUpPropSet(handle,
          (b_puproptype)ParamInfo->proptyp.mgenprop,  /* CONSIDER; check this */
          ParamInfo->defaultval));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestAllPropStore(
    b_handletype handle
)
{
  b_errtype err;
  if (Best16BitRegisterFile(handle))
  {
    /* this is the new style */
    err = BestBasicCommand(handle, CMD_ALL_PROP_STORE, NULL, 0, NULL, NULL);
  }
  else
  {
    /* 8 bit register file */
    b_int8 zw = SAVE_BOARD_CONFIG;
    err = BestBasicBlockWrite(handle, SERVICE_CMD, &zw, 1UL);
  }

  B_ERRETURN(err);
}


b_errtype EXPORT BestAllPropLoad(
    b_handletype handle
)
{
  b_errtype err;
  if (Best16BitRegisterFile(handle))
  {
    /* this is the new style */
    err = BestBasicCommand(handle, CMD_ALL_PROP_LOAD, NULL, 0, NULL, NULL);
  }
  else
  {
    b_int8 zw = RESTORE_BOARD_CONFIG;
    err = BestBasicBlockWrite(handle, SERVICE_CMD, &zw, 1UL);
  }

  B_ERRETURN(err);
}


b_errtype EXPORT BestAllPropDefaultLoad(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* This switch is deliberate 
     * (rather than just checking for an E2925A)...
     * add each hw series case after verifying the correct properties
     */

    switch (BestHwSeriesGet(handle))
    {
    case B_SERIES_E2925A:
      {
        b_int8 zw = RESTORE_BOARD_DEFAULT_CONFIG;
        B_TRY(BestBasicBlockWrite(handle, SERVICE_CMD, &zw, 1UL));
      }
      break;
      
     case B_SERIES_E2924A:	/* host interface */
     case B_SERIES_E2926A:	/* 33 MHz */
     case B_SERIES_E2927A:	/* 66/50 MHz */
     case B_SERIES_E2928A:	/* 66 MHz */
     case B_SERIES_E2940A:	/* 66/50 MHz Compact PCI */
     case B_SERIES_E2925B:	/* replaces the old E2925A */
      {
        b_generic_infotype *decgen;
        b_decodertype dec;
        /* init the internal data base completely this is done in case we
         * cannot do other settings because there might be licenses missing. */
        B_TRY(BestBasicCommand(handle, CMD_INIT_DATABASE,
            NULL, IN_INIT_DATABASE, NULL, NULL));

        B_TRY(BestPowerUpPropSet(handle, B_PU_CONFRESTORE, 1UL));
        /* board properties */

  #if 0
        /* not compilable for customer !! */
        B_TRY(BestDebugPropSet(handle, B_FW_ALL, 0));
  #endif
    
        if (BestCapabilityCheck(handle, B_CAPABILITY_HOSTINT) == B_E_OK)
	{
	  B_TRY(BestMasterGenPropDefaultSet(handle));
	  
	  /* cpuport defaults */
	  B_TRY(BestCPUportPropDefaultSet(handle));
	  
	  /* staticio defaults */
	  B_TRY(BestStaticPropDefaultSet(handle));
	}
	
        /* master attributes */
        /* block properties */

        /* config space */
        if (BestCapabilityCheck(handle, B_CAPABILITY_EXERCISER) == B_E_OK)
        {
	  b_int16 vendorID = 0x103c;
	  b_int16 deviceID;
	  b_int32 register_0;
	  b_hwseriestype HwSeries;
	  
	  /* board attributes */
	  B_TRY(BestBoardPropDefaultSet(handle));


      HwSeries=BestHwSeriesGet(handle);

	  switch (HwSeries)
	  {
	   case B_SERIES_E2924A:	/* host interface */
	    deviceID = 0x2924;
	    break;
	   case B_SERIES_E2925B:	/* new E2925A */
	    deviceID = 0x2925;
	    break;
	   case B_SERIES_E2926A:	/* 33 MHz */
	    deviceID = 0x2926;
	    break;
	   case B_SERIES_E2927A:	/* 66/50 MHz */
	    deviceID = 0x2927;
	    break;
	   case B_SERIES_E2928A:	/* 66 MHz */
	    deviceID = 0x2928;
	    break;
	   case B_SERIES_E2940A:	/* 66/50 MHz Compact PCI */
	    deviceID = 0x2940;
	    break;
	   default:
	    deviceID = 0x2920;	/* generic number */
	  }

	  register_0 = (((b_int32) deviceID) << 16) | vendorID;
	  B_TRY(BestConfRegSet(handle, 0x0UL, register_0));

          B_TRY(BestConfRegMaskSet(handle, 0x0UL, 0x0UL));

          B_TRY(BestConfRegSet(handle, 0x04UL, (HwSeries==B_SERIES_E2928A ? 0x04200000UL /* 66 Mhz capable */ : 0x04000000UL)));
          B_TRY(BestConfRegMaskSet(handle, 0x04UL, 0xf8000317UL));

          B_TRY(BestConfRegSet(handle, 0x08UL, 0xff000000UL)); /* Class Code: Device does not fit... */
          B_TRY(BestConfRegMaskSet(handle, 0x08UL, 0x0UL));

          B_TRY(BestConfRegSet(handle, 0x0cUL, 0x0UL));
          B_TRY(BestConfRegMaskSet(handle, 0x0cUL, 0xffffUL));

          B_TRY(BestConfRegSet(handle, 0x28UL, 0x0UL));
          B_TRY(BestConfRegMaskSet(handle, 0x28UL, 0x0UL));
          B_TRY(BestConfRegSet(handle, 0x2cUL, 0x0UL));
          B_TRY(BestConfRegMaskSet(handle, 0x2cUL, 0x0UL));

          B_TRY(BestConfRegSet(handle, 0x34UL, 0x0UL));
          B_TRY(BestConfRegMaskSet(handle, 0x34UL, 0x0UL));
          B_TRY(BestConfRegSet(handle, 0x38UL, 0x0UL));
          B_TRY(BestConfRegMaskSet(handle, 0x38UL, 0x0UL));
          B_TRY(BestConfRegSet(handle, 0x3cUL, 0x100UL));
          B_TRY(BestConfRegMaskSet(handle, 0x3cUL, 0xffUL));
        }

        if (BestCapabilityCheck(handle, B_CAPABILITY_HOSTINT) == B_E_OK)
        {
	  b_int32 i;
	  b_param_infotype * decparam;
	  
	  /* all decoders */
	  B_TRY(BestGenInfoGet(handle, B_PARAM_DECODER_GEN, &decgen));

	  for (i=0; i<decgen->num_elem; i++)
	  {
	    B_TRY(BestParamInfoGet(handle, B_PARAM_DECODER_GEN,
				   i, &decparam, B_INDEX_SEARCH));

	    dec =  decparam->proptyp.decoder;
	    B_TRY(BestTargetDecoderPropDefaultSet(handle, dec));

	    switch (dec)
	    {
	     case B_DEC_STANDARD_1: /* set to 512K, space64 */
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, 19UL));
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_RESSIZE, 18UL));
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_LOCATION,
					     (b_int32)B_LOC_SPACE64));
	      break;

	     case B_DEC_STANDARD_2: /* grabbed by dec 1 */
	      continue;
	      
	     case B_DEC_STANDARD_3: /* set to I/O, 128 bytes */
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, 7UL));
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_RESSIZE, 7UL));
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_RESBASE, 0x40000UL));
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_LOCATION,
					     (b_int32)B_LOC_IO));
	      break;
	      
	     case B_DEC_CONFIG:	/* leave untouched! */
	      break;
	      
	     default:		/* set rest to size 0 */
	      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, 0UL));
	      break;
	    }                     /*lint !e788 not all enums used */
	    
	    /* TODO: use PUProg */
	    B_TRY(BestTargetDecoderProg(handle, dec));
	  }
	  
          B_TRY(BestTargetGenPropDefaultSet(handle));
        }

        /* observer attributes */
        if (BestCapabilityCheck(handle, B_CAPABILITY_ANALYZER) == B_E_OK)
        {
          B_TRY(BestObsPropDefaultSet(handle));

          /* analyzer defaults */
          B_TRY(BestTracePropDefaultSet(handle));
          /* set the trigger and storage qualifier to default values */
          B_TRY(BestTracePattPropSet(handle, B_PT_TRIGGER, "1"));
          B_TRY(BestTracePattPropSet(handle, B_PT_SQ, "1"));

        }

        /* timing check attributes */
        if (BestCapabilityCheck(handle, B_CAPABILITY_ANALYZER) == B_E_OK)
        {
	  /* CAUTION:
	     Programming the timing checker may fail, because
	     of wrong Busspeed, so we currently 
	     ignore any errors here */
	
	  /* Set mask to default, set B_TCGEN_SPEC to 1, 
	     set defaults in preparation register */
          BestTimCheckDefaultSet(handle);
	  
	  /* program preparation register to HW and clear status */ 
          BestTimCheckProg(handle);

          /* analyzer defaults */
          B_TRY(BestTracePropDefaultSet(handle));
          /* set the trigger and storage qualifier to default values */
          B_TRY(BestTracePattPropSet(handle, B_PT_TRIGGER, "1"));
          B_TRY(BestTracePattPropSet(handle, B_PT_SQ, "1"));
        }

        /* performance defaults */
        /* trigger IO */
        /* board properties */
        B_TRY(BestPowerUpPropDefaultSet(handle));

        /* sst properties */
        if (BestCapabilityCheck(handle, B_CAPABILITY_HOSTINT) == B_E_OK)
        {
	  B_TRY(BestSstPropDefaultSet(handle));
	  B_TRY(BestSstCfgPropDefaultSet(handle));
	}
	
      }
      break;

    default:
      B_TRY_ERROR(B_E_UNKNOWN_HARDWARE);
    }

  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* E2921A_MIN */

/* -----------------------------------------------------------------
 * Miscellaneous Functions
 * ----------------------------------------------------------------- */

#define MAX_VER_STR_LEN   4096

b_errtype EXPORT BestVersionGet(
    b_handletype handle,
    b_versionproptype versionprop,
    b_charptrtype * string
)
{
  B_DECLARE_FUNCNAME("BestVersionGet [vget]");

  b_errtype err;
  b_int32 value;
  static char versionstring[MAX_VER_STR_LEN];
  b_int16 length = (MAX_VER_STR_LEN - 1);
  b_int8 in_zw;
  b_charptrtype pChar;
  *versionstring = 0;

  B_FCT_PARAM_NULL_POINTER_CHK(string);

  /* catch this special case - we have this information at our fingertips... */
  if (versionprop == B_VER_CAPI)
  {
    *string = capi_version_string;
    B_ERRETURN(B_E_OK);
  }

  /* first check if we talk to the new hardware */
  if (Best16BitRegisterFile (handle))  /* new protocol and new hw */
  {
    /* New 2x protocol */
    switch (versionprop)
    {
    case B_VER_SMDATE:
    case B_VER_DPDATE:
    case B_VER_LATTDATE:
    case B_VER_DEEPTRACE:
      *string = " --- ";
      B_ERRETURN(B_E_OK);

    case B_VER_BOARD:
    case B_VER_XILDATE:
    case B_VER_PRODUCT:
    case B_VER_SERIAL:
    case B_VER_FIRMWARE:
    case B_VER_CORE:
    case B_VER_TEAM:
    case B_VER_ICHIBAN:
    case B_VER_FIRMWARE_DATE:
      {
        in_zw = (b_int8) versionprop;
        length = OUT_VERSION_GET;

        err = BestBasicCommand(handle, CMD_VERSION_GET,
          &in_zw, IN_VERSION_GET,
          (b_int8 *) versionstring, &length);
        length = (b_int16) min(length, (MAX_VER_STR_LEN - 1UL));
        versionstring[length] = '\0';
        if (versionprop == B_VER_XILDATE)
          value = strtoul(versionstring, 0, 16);  /* hexadecimal value */
      }
      break;

    case B_VER_CAPI:            /* handled above */
    default:
      B_FCT_PARAM_MSG(2, "Invalid property (versionprop)");
      B_ERRETURN(B_E_FCT_PARAM);
    }                           /* switch */
  }
  else
  {                             /* old version of the version prop property */

    switch (versionprop)
    {
    case B_VER_PRODUCT:
      err = BestBasicBlockRead(handle, PRODUCT_STRING,
        (b_int8 *) versionstring, 20UL);
      break;

    case B_VER_SERIAL:
      err = BestBasicBlockRead(handle, SERIAL_STRING,
        (b_int8 *) versionstring, 20UL);
      break;

    case B_VER_BOARD:
      err = BestBasicBlockRead(handle, REVISION_STRING,
        (b_int8 *) versionstring, 20UL);
      break;

    case B_VER_SMDATE:
      err = BestBasicBlockRead(handle, SM_DATE_CODE, (b_int8 *) & value, 4UL);
      break;

    case B_VER_DPDATE:
      err = BestBasicBlockRead(handle, DP_DATE_CODE, (b_int8 *) & value, 4UL);
      break;

    case B_VER_LATTDATE:
      err = BestBasicBlockRead(handle, LATT_STRING,
        (b_int8 *) versionstring, 20UL);
      break;

    case B_VER_XILDATE:
      err = BestBasicBlockRead(handle, XIL_DATE_CODE, (b_int8 *) & value, 4UL);
      break;

    case B_VER_FIRMWARE:
      err = BestBasicBlockRead(handle, BIOS_STRING,
        (b_int8 *) versionstring, 20UL);
      break;

    case B_VER_CORE:
      err = BestBasicBlockRead(handle, CORE_STRING,
        (b_int8 *) versionstring, 20UL);
      break;

    case B_VER_DEEPTRACE:
      /* read product string ... not deep trace string */
      if (B_E_OK != (err = BestBasicBlockRead(handle, PRODUCT_STRING,
            (b_int8 *) versionstring, 20UL)))
        B_ERRETURN(err);

      if (strcmp(versionstring, "E2925A_DEEP") == 0)
        *string = versionstring;
      else
        *string = " --- ";

      B_ERRETURN(err);

    case B_VER_TEAM:
    case B_VER_ICHIBAN:
    case B_VER_FIRMWARE_DATE:
      *string = " --- ";
      B_ERRETURN(B_E_OK);

    case B_VER_CAPI:            /* handled above */
    default:
      B_FCT_PARAM_MSG(2, "Invalid property (versionprop)");
      B_ERRETURN(B_E_FCT_PARAM);
    }                           /*lint !e788 */
  }

  /* now convert string, if necessary */
  switch (versionprop)
  {
  case B_VER_SMDATE:
  case B_VER_DPDATE:
  case B_VER_XILDATE:
    sprintf(versionstring, "%s", ctime((time_t *) & value));
    if (NULL != (pChar = strchr(versionstring, '\n')))
    {
      *pChar = '\0';            /* get rid of '\n' */
    }
  }                             /*lint !e787 */

  *string = versionstring;

  B_ERRETURN(err);
}



b_errtype EXPORT BestSMReset(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_SM_RESET,
          NULL, IN_SM_RESET, NULL, NULL));
    }

    else
    {
      b_int16 val = PCI_RESET_SM;
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8 *) & val, 2UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}



b_errtype EXPORT BestBoardReset(b_handletype handle)
{

#define WAIT_RESET     (b_int32)1000     /* we wait this long after disconnecting */
#define WAIT_RECONNECT (b_int32)5000     /* we'll wait this long for a reconnect */
#define BOOT_BAUD_E2925  9600

  B_TRY_VARS_NO_PROG;
  BEST_TIMER tmrTotal;
  HANDLECHECK;

  B_TRY_BEGIN
  {
    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_BOARD_RESET,
          NULL, IN_BOARD_RESET, NULL, NULL));
    }
    else
    {
      b_porttype port;
      b_int32 portnum;
      b_int32 baud;
      B_TRY(BestGetPortFromHandle(handle, &port));
      B_TRY(BestGetPortnumberFromHandle(handle, &portnum));
      baud = handle_array[handle].param;

      /* a serial port can only do this at the default baud rate */
      if (port == B_PORT_RS232 && baud != BOOT_BAUD_E2925)
      {
        B_TRY(BestRS232BaudRateSet(handle, BOOT_BAUD_E2925));
      }

      /* core mode or not ? */
      if( BestIs2925(handle) )
      {
        b_int16 val = SYS_RESET;

        B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8 *) & val, 2UL));

        /* disconnect */
        (void) BestDisconnect(handle);

        /* wait a while */
        BestStartTimeout(WAIT_RESET, &tmrTotal);
        while(!BestIsTimeoutDone(&tmrTotal)) {;}

        /* give the board a while to reset */
        BestStartTimeout(WAIT_RECONNECT, &tmrTotal);

        while (B_E_OK != BestConnect(handle))
        {
          B_TRY_FAIL(BestIsTimeoutDone(&tmrTotal) ? B_E_BOARD_RESET : B_E_OK);
        }
      }
      else
      {
        b_int32 open_tries = RETRY_COUNT_AFTER_BOOT; 
        b_errtype open_err;
        b_int8 val = 'r';

        B_TRY_FAIL(BestIsCore(handle) ? B_E_OK : B_E_BAD_HANDLE);

        /* reset board, ignore return value, CZ */
        (void) BestBasicBlockWrite(handle, BOARD_RESET, &val, 1UL);

        BestStartTimeout((b_int32)WAIT_RECONNECT, &tmrTotal);

        while (B_E_OK == BestCheckConnection(handle))
        {
          B_TRY_FAIL(BestIsTimeoutDone(&tmrTotal) ? B_E_ERROR : B_E_OK);
        }

        (void) BestCloseAdv(handle, B_CLOSE_RESERVED);

        do
        {
          open_err = BestOpenAdv(&handle, port, portnum,
            NULL, B_OPEN_STANDARD_RESERVED);

        } while ((open_err != B_E_OK) && --open_tries);

        B_TRY_FAIL(open_err);

      }

      /* reset baudrate if we changed it */
      if (port == B_PORT_RS232 && baud != BOOT_BAUD_E2925)
      {
        B_TRY(BestRS232BaudRateSet(handle, baud));
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestBoardRestart(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    if (BestIsCore(handle))
    {
      B_TRY(BestBoardReset(handle));
    }
    else
    {
      B_TRY(BestBasicCommand(handle, CMD_BOARD_RESTART,
          NULL, IN_BOARD_RESTART, NULL, NULL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestCapiPropSet(
    b_handletype handle,
    b_capiproptype capiprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestCapiPropSet [capiprpset]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN {
    /* no capability checking */
    B_TRY_FCT_PARAM (capiprop, capiprop >= B_CAPIPROP_SIZE);
    best_capi_prop [handle] [capiprop] = value;
  }
  
  B_ERRETURN (B_TRY_RET);
}


b_errtype EXPORT BestCapiPropGet(
    b_handletype handle,
    b_capiproptype capiprop,
    b_int32 *value
)
{
  B_DECLARE_FUNCNAME("BestCapiPropGet [capiprpget]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN {
    /* no capability checking */
    B_TRY_FCT_PARAM (capiprop, capiprop >= B_CAPIPROP_SIZE);
    B_TRY_FCT_PARAM_NULL_POINTER (value);
    *value = best_capi_prop [handle] [capiprop];
  }
  
  B_ERRETURN (B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * PCI Exerciser Generic Property Programming Functions
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestExerciserGenPropGet(b_handletype handle,
    b_exercisergenproptype exegenprop,
    b_int32 * value
)
{

  B_DECLARE_FUNCNAME("BestExerciserGenPropGet [egprpget]");

  B_TRY_VARS_NO_PROG;

  b_param_infotype *ParamInfo;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);
    B_TRY_FCT_PARAM_NULL_POINTER(value);
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    /* Get pointer to i-th (existing) property */
    B_TRY(BestParamInfoGet(handle, B_PARAM_EXERCISER_GEN,
        (b_int32)exegenprop, &ParamInfo, (b_int32)B_ENUM_SEARCH));
    /* Get current value */
    B_TRY(BestAbstractPropLongGet(handle, CMD_EXE_EGEN_PROP_GET,
        (b_int8) ParamInfo->proptyp.egenprop, value));
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestExerciserGenPropSet(b_handletype handle,
    b_exercisergenproptype exegenprop,
    b_int32 value)

{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* dynamic capability checking concerning values of parameters */
    B_TRY(BestParamCheck(handle, B_PARAM_EXERCISER_GEN, (b_int32)exegenprop, value));
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    B_TRY(BestAbstractPropLongSet(handle, CMD_EXE_EGEN_PROP_SET,
        (b_int8) exegenprop, value));
  }

  B_ERRETURN(B_TRY_RET);
}
